/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx_arch_h_
#define _mx_arch_h_

#include <sys/cred.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <inttypes.h>
#include <sys/conf.h>
#include <sys/file.h>
#include <sys/modctl.h>
#include <sys/open.h>
#include <sys/stat.h>
#include <sys/ddi.h>
#include <sys/sunddi.h>
#include <sys/esunddi.h>
#include <sys/sysmacros.h>
#include <sys/atomic.h>

/* Get rid of annoying definition of "u" to (curproc->p_user) */
#ifdef u
#undef u
#endif

#include "mx_auto_config.h"

/* Solaris doesn't define PAGE_SIZE, but a few places 
   depend on it */
#if  MX_CPU_x86_64 ||MX_CPU_x86
#define PAGE_SIZE 4096
#define MX_SOLARIS_ALWAYS_CONSISTENT 0
#elif defined(MX_CPU_sparc64)
/* #error we have not tested on sparc */
#define PAGE_SIZE 8192
#define MX_SOLARIS_ALWAYS_CONSISTENT 1
#endif

#include "bsd/queue.h"
#include "subr_scanf.h"
#include "mx_int.h"
#include "mx_io_impl.h"
#include "mcp_global.h"
#include "myriexpress.h"

void mx_solaris_info(char *format, ...);
void mx_solaris_warn(char *format, ...);

#define MX_PRINT(s) printf s
#define MX_WARN(s) do {                 \
  mx_solaris_warn s;                    \
} while (0)

#define MX_INFO(s) do {                 \
  mx_solaris_info s;                    \
} while (0)

#include "mx_debug.h"

struct mx_io_mapping
{
  void *kaddr;			/* address of mapping */
  ddi_acc_handle_t handle;
  uint32_t offset;
  uint32_t len;
};


struct mx_instance_state;
typedef void (ether_tx_done_t)(struct mx_instance_state *, uint32_t);
typedef void (ether_rx_done_t)(struct mx_instance_state *, uint32_t, uint32_t, 
			       uint32_t, uint32_t);
typedef void (ether_link_change_t)(struct mx_instance_state *);


#define MX_ARCH_MAX_MAPPINGS 4

typedef struct mx_arch_instance_info
{
  dev_info_t *dip;		/* Pointer to DDI/DDK device info */
  ddi_acc_handle_t pci_acc_handle; /* for accessing PCI configuration space. */

  /* interrupt state */
  ddi_iblock_cookie_t iblock_cookie;
  ddi_idevice_cookie_t idevice_cookie;
#if MX_SOLARIS_COMPLICATED_SPIN
  mx_sync_t spin_sync;
#endif
  int reg_set;			/* The register set used to map the board. */
  char reg_set_cached;		/* boolean indicating if reg_set is cached */
  char board_span_cached;	/* boolean indicating if board_span cached */
  struct mx_io_mapping	mappings[MX_ARCH_MAX_MAPPINGS];

  /* ethernet hooks */
  ether_tx_done_t *ether_tx_done;
  ether_rx_done_t *ether_rx_done_small;
  ether_rx_done_t *ether_rx_done_big;
  ether_link_change_t *ether_link_change;
} mx_arch_instance_info_t;

typedef struct mx_arch_endpt_info
{
  struct proc *proc;
} mx_arch_endpt_info_t;

#define MX_SOLARIS_IS_MASTER(es) (es->arch.master_es == 0)
#define MX_SOLARIS_IS_SLAVE(es) (!MX_SOLARIS_IS_MASTER (es))

#define MX_ARCH_SYNC_TAG 0x12AB34CD
typedef struct mx_sync
{
  /* mutex, which will eventually be separated out */
  kmutex_t mu;

  /* fields used to implement a semaphore */
  int wake_cnt;
  kmutex_t _mu;
  kcondvar_t cv;

  /* a tag for marking this struct */
  unsigned int tag;
} mx_sync_t;

typedef kmutex_t mx_spinlock_t;

struct mx_page_pin
{
  mcp_dma_addr_t dma;
  uint64_t va;
  ddi_dma_handle_t dma_handle;
  ddi_umem_cookie_t umem_cookie;
  ddi_dma_cookie_t dma_cookie;     /* can be deleted in mx_alloc_dma_page */
  unsigned int sts;
};

#define MX_OPTIMIZED_DMA_PAGE_ALLOC 1
#define MX_FACTORIZED_PAGE_PIN 1

struct mx_instance_state;
struct mx_copyblock;

int
mx_optimized_alloc_copyblock(struct mx_instance_state *is,
			     struct mx_copyblock *cb);
void
mx_optimized_free_copyblock(struct mx_instance_state *is,
			    struct mx_copyblock *cb);

#define mx_alloc_copyblock(is,cb) mx_optimized_alloc_copyblock(is,cb)
#define mx_free_copyblock(is,cb)  mx_optimized_free_copyblock(is,cb)

/* Copy user data to a kernel buffer. */
static inline int
mx_arch_copyin (mx_uaddr_t what, void *where, size_t amount)
{
  int errno;

  errno = ddi_copyin ((void *) what, where, amount, 0);
  if (errno) {
    return EFAULT;
  }
  return 0;
}

/* Copy kernel data to a user buffer */
static inline int
mx_arch_copyout (const void *what, mx_uaddr_t where, size_t amount)
{
  int errno;

  errno = ddi_copyout (what, (void *) where, amount, 0);
  if (errno) {
    return EFAULT;
  }
  return 0;
}

void mx_spin(uint32_t);

/* Solaris normalizations */

#ifndef MIN
#define MIN(a,b) (((a)<(b))?(a):(b))
#endif

#ifndef MAX
#define MAX(a,b) (((a)>(b))?(a):(b))
#endif
 
/* 64-bit byeswapping, in case we need it */

static inline uint64_t
swap_64(uint64_t val)
{
  return (((uint64_t)ntohl((uint32_t)val) << 32) 
	  + ntohl(val >> 32));
}

#if defined (_BIG_ENDIAN)
#define mx_hton_u64(x) (x)
#define mx_ntoh_u64(x) (x)
#else
#define mx_hton_u64(x) swap_64(x)
#define mx_ntoh_u64(x) swap_64(x)
#endif

/* atomic types and operations */
/* Before solaris 10, atomic operations are only available in KERNEL space.
 * Solaris 10 starts to provide atomic operations in user space */
typedef uint32_t                        mx_atomic_t;
#define mx_atomic_add(value, ptr)       atomic_add_32(ptr, value)
#define mx_atomic_subtract(value, ptr)  atomic_add_32(ptr, -value)
#define mx_atomic_read(ptr)             atomic_add_32_nv(ptr, 0)
#define mx_atomic_set(ptr, value)       *(ptr) = (value)


#define mx_mutex_enter(s)   mutex_enter(&(s)->mu)
#define mx_mutex_exit(s)    mutex_exit(&(s)->mu)

/* spinlocks not required on solaris because the interrupt handler
   can take a mutex */

#define mx_spin_lock(lock)    			mutex_enter(lock)
#define mx_spin_unlock(lock)  		 	mutex_exit(lock)
#define mx_spin_lock_irqsave(lock, flags)     	mutex_enter(lock)
#define mx_spin_unlock_irqrestore(lock, flags)  mutex_exit(lock)

static inline int
mx_arch_memcpy_from_segment(void *ptr, uint64_t segment_ptr,
			    size_t amount, uintptr_t memory_context)
{
  return EINVAL;
}

static inline int
mx_arch_memcpy_to_segment(uint64_t segment_ptr, void *ptr,
			  size_t amount, uintptr_t memory_context)
{
  return EINVAL;
}

#define mx_reserve_page(x)
#define mx_unreserve_page(x)
#define mx_get_memory_context() 0

/* memory bar definition */
#if defined (__GNUC__)
  #define likely(x)       __builtin_expect((x),1)
  #define unlikely(x)     __builtin_expect((x),0)

  #if #cpu(i386) || defined __i386 || defined i386 || defined __i386__ || #cpu(x86_64) || defined __x86_64__
    #define mb()  __asm__ __volatile__ ("sfence;": : :"memory")
  #elif #cpu(sparc64) || defined sparc64 || defined __sparcv9 
    #define mb()  __asm__ __volatile__ ("membar #MemIssue": : :"memory")
  #elif #cpu(sparc) || defined sparc || defined __sparc__
    #define mb()  __asm__ __volatile__ ("stbar;": : :"memory")
  #endif
#elif defined  (__SUNPRO_C) || defined(__SUNPRO_CC)
  void mb(void);
  #define likely(x)       (x)
  #define unlikely(x)     (x)
#else
  #error "unknown compiler"
#endif

#endif /* _mx_arch_h_ */
